<?php
/* vim: set expandtab tabstop=4 softtabstop=4 shiftwidth=4:
  Codificación: UTF-8
  +----------------------------------------------------------------------+
  | Issabel version 0.5                                                  |
  | http://www.issabel.org                                               |
  +----------------------------------------------------------------------+
  | Copyright (c) 2006 Palosanto Solutions S. A.                         |
  +----------------------------------------------------------------------+
  | The contents of this file are subject to the General Public License  |
  | (GPL) Version 2 (the "License"); you may not use this file except in |
  | compliance with the License. You may obtain a copy of the License at |
  | http://www.opensource.org/licenses/gpl-license.php                   |
  |                                                                      |
  | Software distributed under the License is distributed on an "AS IS"  |
  | basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See  |
  | the License for the specific language governing rights and           |
  | limitations under the License.                                       |
  +----------------------------------------------------------------------+
  | The Initial Developer of the Original Code is PaloSanto Solutions    |
  +----------------------------------------------------------------------+
  $Id: paloSantoCampaignCC.class.php,v 1.2 2008/06/06 07:15:07 cbarcos Exp $ */

include_once("libs/paloSantoDB.class.php");

define('REGEXP_FECHA_VALIDA', '/^\d{4}-\d{2}-\d{2}$/');
define('REGEXP_HORA_VALIDA', '/^\d{2}:\d{2}$/');

/* Clase que implementa campaña (saliente por ahora) de CallCenter (CC) */
class paloSantoCampaignCC
{
    var $_DB; // instancia de la clase paloDB
    var $errMsg;

    function paloSantoCampaignCC(&$pDB)
    {
        // Se recibe como parámetro una referencia a una conexión paloDB
        if (is_object($pDB)) {
            $this->_DB =& $pDB;
            $this->errMsg = $this->_DB->errMsg;
        } else {
            $dsn = (string)$pDB;
            $this->_DB = new paloDB($dsn);

            if (!$this->_DB->connStatus) {
                $this->errMsg = $this->_DB->errMsg;
                // debo llenar alguna variable de error
            } else {
                // debo llenar alguna variable de error
            }
        }
    }

    /**
     * Procedimiento para obtener el listado de los campañas existentes. Si
     * se especifica id, el listado contendrá únicamente la campaña
     * indicada por el valor. De otro modo, se listarán todas las campañas.
     *
     * @param int   $id_campaign    Si != NULL, indica el id de la campaña a recoger
     *
     * @return array    Listado de campañas en el siguiente formato, o FALSE en caso de error:
     *  array(
     *      //array(id,nombre,fecha_ini,hora_ini,prompt,llamadas_prog,llamadas_real,reintentos,llamadas_pend,detalles),
     *		array(id, name, start_time, retries, b_status, trunk),
     *      ...
     *  )
     */
    function getCampaigns($limit, $offset, $id_campaign = NULL, $estatus = 'all') {
        
        $this->errMsg = '';
        if (!is_null($id_campaign) && !ctype_digit("$id_campaign")) {
            $this->errMsg = _tr("Campaign ID is not valid");
            return FALSE;
        }
        $sPeticionSQL = <<<SQL_SELECT_CAMPAIGNS
SELECT id, name, datetime_init, datetime_end, daytime_init,
    daytime_end,  retries,  num_completadas, estatus, id_num_keyboard, id_sounds, max_canales, breaktime_init, breaktime_end, active_day 
FROM campaign 
SQL_SELECT_CAMPAIGNS;
        $paramWhere = array();
        $paramSQL = array();

        if (in_array($estatus, array('A', 'I', 'T'))) {
            $paramWhere[] = 'estatus = ?';
            $paramSQL[] = $estatus;
        }
        if (!is_null($id_campaign)) {
            $paramWhere[] = 'id = ?';
            $paramSQL[] = $id_campaign;
        }
        if (count($paramWhere) > 0)
            $sPeticionSQL .= ' WHERE ' . implode(' AND ', $paramWhere);
        //$sPeticionSQL .= ' ORDER BY datetime_init, daytime_init';
	$sPeticionSQL .= ' ORDER BY id';
        if (!is_null($limit)) {
            $sPeticionSQL .= ' LIMIT ? OFFSET ?';
            $paramSQL[] = $limit;
            $paramSQL[] = $offset;
        }
        $arr_result = $this->_DB->fetchTable($sPeticionSQL, TRUE, $paramSQL);
	if (!is_array($arr_result)) {  
            $this->errMsg = $this->_DB->errMsg;
            return FALSE;
        } 
        return $arr_result;
    }
    
    function getSounds() { 
        
        $sPeticionSQL = <<<SQL_SELECT_CAMPAIGNS
SELECT id, name
FROM audios
SQL_SELECT_CAMPAIGNS;

        $paramSQL = array();
        
          $arr_result = $this->_DB->fetchTable($sPeticionSQL, TRUE, $paramSQL);
        if (!is_array($arr_result)) { 
            $this->errMsg = $this->_DB->errMsg;
            return FALSE;
        } 
        return $arr_result;
        
    }

    /**
     * Procedimiento para crear una nueva campaña, vacía e inactiva. Esta campaña
     * debe luego llenarse con números de teléfono en sucesivas operaciones.
     *
     * @param   $sNombre            Nombre de la campaña
     * @param   $iMaxCanales        Número máximo de canales a usar simultáneamente por campaña
     * @param   $iRetries           Número de reintentos de la campaña, por omisión 5
     * @param   $sTrunk             troncal por donde se van a realizar las llamadas (p.ej. "Zap/g0")
     * @param   $sContext           Contexto asociado a la campaña (p.ej. 'from-internal')
     * @param   $sQueue             Número que identifica a la cola a conectar la campaña saliente (p.ej. '402')
     * @param   $sFechaInicio       Fecha YYYY-MM-DD en que inicia la campaña
     * @param   $sFechaFinal        Fecha YYYY-MM-DD en que finaliza la campaña
     * @param   $sHoraInicio        Hora del día (HH:MM militar) en que se puede iniciar llamadas
     * @param   $sHoraFinal         Hora del día (HH:MM militar) en que se debe dejar de hacer llamadas
     * @param   $script             Texto del script a recitar por el agente
     * @param   $id_url             NULL, o ID del URL externo a cargar
     *
     * @return  int    El ID de la campaña recién creada, o NULL en caso de error
     */
    function createEmptyCampaign($sNombre, $iMaxCanales, $iRetries, $sTrunk, $sContext, $sQueue,
        $sFechaInicial, $sFechaFinal, $sHoraInicio, $sHoraFinal, $script, $id_url)
    {
        $id_campaign = NULL;
        $bExito = FALSE;

        // Carga de colas entrantes activas
        $recordset = $this->_DB->fetchTable("SELECT queue FROM queue_call_entry WHERE estatus='A'");
        if (!is_array($recordset)) {
            $this->errMsg = _tr('(internal) Failed to query active incoming queues').
                ' - '.$this->_DB->errMsg;
        	return NULL;
        }
        $colasEntrantes = array();
        foreach ($recordset as $tupla) $colasEntrantes[] = $tupla[0];

        $sNombre = trim($sNombre);
        $iMaxCanales = trim($iMaxCanales);
        $iRetries = trim($iRetries);
        $sTrunk = trim($sTrunk);
        $sContext = trim($sContext);
        $sQueue = trim($sQueue);
        $sFechaInicial = trim($sFechaInicial);
        $sFechaFinal = trim($sFechaFinal);
        $sHoraInicio = trim($sHoraInicio);
        $sHoraFinal = trim($sHoraFinal);
        $script = trim($script);

        if ($sTrunk == '') $sTrunk = NULL;

        if ($sNombre == '') {
            $this->errMsg = _tr("Name Campaign can't be empty");//'Nombre de campaña no puede estar vacío';
        } elseif ($sContext == '') {
            $this->errMsg = _tr("Context can't be empty");//'Contexto no puede estar vacío';
        } elseif (!ctype_digit($iRetries)) {
            $this->errMsg = _tr('Retries must be numeric');//'Número de reintentos debe de ser numérico y entero';
        } elseif ($sQueue == '') {
            $this->errMsg = _tr("Queue can't be empty");//'Número de cola no puede estar vacío';
        } elseif (!ctype_digit($sQueue)) {
            $this->errMsg = _tr('Queue must be numeric');//'Número de cola debe de ser numérico y entero';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaInicial)) {
            $this->errMsg = _tr('Invalid Start Date');//'Fecha de inicio no es válida (se espera yyyy-mm-dd)';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaFinal)) {
            $this->errMsg = _tr('Invalid End Date');//'Fecha de final no es válida (se espera yyyy-mm-dd)';
        } elseif ($sFechaInicial > $sFechaFinal) {
            $this->errMsg = _tr('Start Date must be greater than End Date');//'Fecha de inicio debe ser anterior a la fecha final';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraInicio)) {
            $this->errMsg = _tr('Invalid Start Time');//'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraFinal)) {
            $this->errMsg = _tr('Invalid End Time');//'Hora de final no es válida (se espera hh:mm)';
        } elseif (strcmp($sFechaInicial,$sFechaFinal)==0 && strcmp ($sHoraInicio,$sHoraFinal)>=0) {
            $this->errMsg = _tr('Start Time must be greater than End Time');//'Hora de inicio debe ser anterior a la hora final';
        } elseif (!is_null($id_url) && !ctype_digit("$id_url")) {
            $this->errMsg = _tr('(internal) Invalid URL ID');
        } elseif (in_array($sQueue, $colasEntrantes)) {
             $this->errMsg =  _tr('Queue is being used, choose other one');//La cola ya está siendo usada, escoja otra
        } else {
            // Verificar que el nombre de la campaña es único
            $tupla = $this->_DB->getFirstRowQuery(
                'SELECT COUNT(*) AS N FROM campaign WHERE name = ?', TRUE, array($sNombre));
            if (is_array($tupla) && $tupla['N'] > 0) {
                // Ya existe una campaña duplicada
                $this->errMsg = _tr('Name Campaign already exists');//'Nombre de campaña indicado ya está en uso';
            	return NULL;
            }

            // Construir y ejecutar la orden de inserción SQL
            $sPeticionSQL = <<<SQL_INSERT_CAMPAIGN
INSERT INTO campaign (name, max_canales, retries, trunk, context, queue,
    datetime_init, datetime_end, daytime_init, daytime_end, script, id_url)
VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
SQL_INSERT_CAMPAIGN;
            $paramSQL = array($sNombre, $iMaxCanales, $iRetries, $sTrunk,
                $sContext, $sQueue, $sFechaInicial, $sFechaFinal, $sHoraInicio,
                $sHoraFinal, $script, $id_url);
            if ($this->_DB->genQuery($sPeticionSQL, $paramSQL)) {
            	// Leer el ID insertado por la operación
                $id_campaign = $this->_DB->getLastInsertId();
                if ($id_campaign === FALSE) {
                	$this->errMsg = $this->_DB->errMsg;
                    $id_campaign = NULL;
                }
            } else {
            	$this->errMsg = $this->_DB->errMsg;
            }
        }
        return $id_campaign;
    }
    
        function createEmptyCampaignNew($sNombre, $iRetries, $sContext, $sFechaInicial, $sFechaFinal, $sHoraInicio, $sHoraFinal, $id_num_keyboard, $id_sounds, $max_canales, $break_ini, $break_fin, $active_days = array()) {
            
        $id_campaign = NULL;
        $bExito = FALSE;

        // Carga de colas entrantes activas
        /*
        $recordset = $this->_DB->fetchTable("SELECT queue FROM queue_call_entry WHERE estatus='A'");
        if (!is_array($recordset)) {
            $this->errMsg = _tr('(internal) Failed to query active incoming queues') .
                    ' - ' . $this->_DB->errMsg;
            return NULL;
        }
        
        $colasEntrantes = array();
        foreach ($recordset as $tupla)
            $colasEntrantes[] = $tupla[0];
        */
        
        $sNombre = trim($sNombre);
        //$iMaxCanales = trim($iMaxCanales);
        $iRetries = trim($iRetries);
       // $sTrunk = trim($sTrunk);
        //$sContext = trim($sContext);
        //$sQueue = trim($sQueue);
        $sFechaInicial = trim($sFechaInicial);
        $sFechaFinal = trim($sFechaFinal);
        $sHoraInicio = trim($sHoraInicio);
        $sHoraFinal = trim($sHoraFinal);
        
        $break_in = trim($break_in);
        $break_fin = trim($break_fin);
       // $script = trim($script);
        /*
        if ($sTrunk == '')
            $sTrunk = NULL;
        */

        if ($sNombre == '') {
            $this->errMsg = _tr("Name Campaign can't be empty"); //'Nombre de campaña no puede estar vacío';
       // } elseif ($sContext == '') {
        //    $this->errMsg = _tr("Context can't be empty"); //'Contexto no puede estar vacío';
        } elseif (!ctype_digit($iRetries)) {
            $this->errMsg = _tr('Retries must be numeric'); //'Número de reintentos debe de ser numérico y entero';
        //} elseif ($sQueue == '') {
         //   $this->errMsg = _tr("Queue can't be empty"); //'Número de cola no puede estar vacío';
       // } elseif (!ctype_digit($sQueue)) {
        //    $this->errMsg = _tr('Queue must be numeric'); //'Número de cola debe de ser numérico y entero';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaInicial)) {
            $this->errMsg = _tr('Invalid Start Date'); //'Fecha de inicio no es válida (se espera yyyy-mm-dd)';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaFinal)) {
            $this->errMsg = _tr('Invalid End Date'); //'Fecha de final no es válida (se espera yyyy-mm-dd)';
        } elseif ($sFechaInicial > $sFechaFinal) {
            $this->errMsg = _tr('Start Date must be greater than End Date'); //'Fecha de inicio debe ser anterior a la fecha final';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraInicio)) {
            $this->errMsg = _tr('Invalid Start Time'); //'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraFinal)) {
            $this->errMsg = _tr('Invalid End Time'); //'Hora de final no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $break_ini)) {
            $this->errMsg = _tr('Invalid Start Time'); //'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $break_fin)) {
            $this->errMsg = _tr('Invalid End Time');
        }elseif (strcmp($sFechaInicial, $sFechaFinal) == 0 && strcmp($sHoraInicio, $sHoraFinal) >= 0 && strcmp($break_ini, $break_fin) >= 0) {
            $this->errMsg = _tr('Start Time must be greater than End Time'); //'Hora de inicio debe ser anterior a la hora final';
       // } elseif (!is_null($id_url) && !ctype_digit("$id_url")) {
        ///    $this->errMsg = _tr('(internal) Invalid URL ID');
        //} elseif (in_array($sQueue, $colasEntrantes)) {
          //  $this->errMsg = _tr('Queue is being used, choose other one'); //La cola ya está siendo usada, escoja otra
        } else {
            // Verificar que el nombre de la campaña es único
            $tupla = $this->_DB->getFirstRowQuery(
                    'SELECT COUNT(*) AS N FROM campaign WHERE name = ?', TRUE, array($sNombre));
            if (is_array($tupla) && $tupla['N'] > 0) {
                // Ya existe una campaña duplicada
                $this->errMsg = _tr('Name Campaign already exists'); //'Nombre de campaña indicado ya está en uso';
                return NULL;
            }

            // Construir y ejecutar la orden de inserción SQL
            $sPeticionSQL = <<<SQL_INSERT_CAMPAIGN
INSERT INTO campaign (name, retries, context, datetime_init, datetime_end, daytime_init, daytime_end, id_num_keyboard, id_sounds, max_canales, breaktime_init, breaktime_end, active_day, estatus)
VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
SQL_INSERT_CAMPAIGN;
            $paramSQL = array($sNombre, $iRetries, $sContext, $sFechaInicial, $sFechaFinal, $sHoraInicio, $sHoraFinal, $id_num_keyboard, $id_sounds, $max_canales, $break_ini, $break_fin, json_encode($active_days), 'I');
            if ($this->_DB->genQuery($sPeticionSQL, $paramSQL)) {
                // Leer el ID insertado por la operación
                $id_campaign = $this->_DB->getLastInsertId();
                if ($id_campaign === FALSE) {
                    $this->errMsg = $this->_DB->errMsg;
                    $id_campaign = NULL;
                }
            } else {
                $this->errMsg = $this->_DB->errMsg;
            }
        }
        return $id_campaign;
    }

    /**
	 * Procedimiento para agregar los formularios a la campaña
	 *
     * @param	int		$id_campaign	ID de la campaña
     * @param	string		$formularios	los id de los formularios 1,2,.....,
     * @return	bool            true or false
    */
    function addCampaignForm($id_campania,$formularios)
    {
        if (!is_array($formularios)) {
            if ($formularios == '')
                $formularios = array();
            else $formularios = explode(',', $formularios);
        }
        foreach ($formularios as $id_form) {
        	$r = $this->_DB->genQuery(
                'INSERT INTO campaign_form (id_campaign, id_form) VALUES (?, ?)',
                array($id_campania, $id_form));
            if (!$r) {
                $this->errMsg = $this->_DB->errMsg;
            	return FALSE;
            }
        }
        return TRUE;
    }

    /**
	 * Procedimiento para actualizar los formularios a la campaña
	 *
     * @param	int		$id_campaign	ID de la campaña
     * @param	string		$formularios	los id de los formularios 1,2,.....,
     * @return	bool            true or false
    */
    function updateCampaignForm($id_campania, $formularios)
    {
        if (!$this->_DB->genQuery(
            'DELETE FROM campaign_form WHERE id_campaign = ?',
            array($id_campania))) {
            $this->errMsg = $this->_DB->errMsg;
            return FALSE;
        }
        return $this->addCampaignForm($id_campania, $formularios);
    }

    /**
	 * Procedimiento para obtener los formualarios de una campaña
	 *
     * @param	int		$id_campaign	ID de la campaña
     * @return	mixed	NULL en caso de error o los id formularios
    */
    function obtenerCampaignForm($id_campania)
    {
        $tupla = $this->_DB->fetchTable(
            'SELECT id_form FROM campaign_form WHERE id_campaign = ?',
            FALSE, array($id_campania));
        if (!is_array($tupla)) {
            $this->errMsg = $this->_DB->errMsg;
            return NULL;
        }
        $salida = array();
        foreach ($tupla as $value) $salida[] = $value[0];
        return $salida;
    }

    function getExternalUrls()
    {
        $tupla = $this->_DB->fetchTable('SELECT id, description FROM campaign_external_url WHERE active = 1');
        if (!is_array($tupla)) {
            $this->errMsg = $this->_DB->errMsg;
            return NULL;
        }
        $salida = array();
        foreach ($tupla as $value) $salida[$value[0]] = $value[1];
        return $salida;
    }

	/**
	 * Procedimiento para contar el número de teléfonos asignados a ser marcados
	 * en la campaña indicada por $idCampaign.
	 *
     * @param	int		$idCampaign	ID de la campaña a leer
     *
     * @return	mixed	NULL en caso de error o número de teléfonos total
	 */
    function countCampaignNumbers($idCampaign)
    {
    	$iNumTelefonos = NULL;

        if (!ctype_digit($idCampaign)) {
            $this->errMsg = _tr('Invalid Campaign ID'); //;'ID de campaña no es numérico';
            return NULL;
        }
        $tupla = $this->_DB->getFirstRowQuery(
            'SELECT COUNT(*) FROM calls WHERE id_campaign = ?',
            FALSE, array($idCampaign));
        if (!is_array($tupla)) {
            $this->errMsg = $this->_DB->errMsg;
            return NULL;
        }
        return (int)$tupla[0];
    }

    function countCampaignNumbersTotal($idCampaign)
    {
    	$iNumTelefonos = NULL;

        if (!ctype_digit($idCampaign)) {
            $this->errMsg = _tr('Invalid Campaign ID'); //;'ID de campaña no es numérico';
            return NULL;
        }
        $tupla = $this->_DB->getFirstRowQuery(
            'SELECT COUNT(*) FROM calls WHERE id_campaign = ?',
            FALSE, array($idCampaign));
        if (!is_array($tupla)) {
            $this->errMsg = $this->_DB->errMsg;
            return NULL;
        }
        return (int)$tupla[0];
    }
    
    function countCampaignNumbersComplete($idCampaign)
    {
    	$iNumTelefonos = NULL;

        if (!ctype_digit($idCampaign)) {
            $this->errMsg = _tr('Invalid Campaign ID'); //;'ID de campaña no es numérico';
            return NULL;
        }
        $tupla = $this->_DB->getFirstRowQuery(
            'SELECT COUNT(*) FROM calls WHERE id_campaign = ? AND status IS NOT NULL',
            FALSE, array($idCampaign));
        if (!is_array($tupla)) {
            $this->errMsg = $this->_DB->errMsg;
            return NULL;
        }
        return (int)$tupla[0];
    }

    /**
     * Procedimiento para modificar las propiedades de una campaña existente
     *
     * @param   $idCampaign         ID de la campaña existente
     * @param   $sNombre            Nombre de la campaña
     * @param   $iMaxCanales        Número máximo de canales a usar simultáneamente por campaña
     * @param   $iRetries           Número de reintentos de la campaña, por omisión 5
     * @param   $sTrunk             troncal por donde se van a realizar las llamadas (p.ej. "Zap/g0")
     * @param   $sContext           Contexto asociado a la campaña (p.ej. 'from-internal')
     * @param   $sQueue             Número que identifica a la cola a conectar la campaña saliente (p.ej. '402')
     * @param   $sFechaInicio       Fecha YYYY-MM-DD en que inicia la campaña
     * @param   $sFechaFinal        Fecha YYYY-MM-DD en que finaliza la campaña
     * @param   $sHoraInicio        Hora del día (HH:MM militar) en que se puede iniciar llamadas
     * @param   $sHoraFinal         Hora del día (HH:MM militar) en que se debe dejar de hacer llamadas
     * @param   $script             Texto del script a recitar por el agente
     * @param   $id_url             NULL, o ID del URL externo a cargar
     *
     * @return  bool                VERDADERO si se actualiza correctamente, FALSO en error
     */
    function updateCampaign($idCampaign, $sNombre, $iMaxCanales, $iRetries, $sTrunk,
        $sContext, $sQueue, $sFechaInicial, $sFechaFinal, $sHoraInicio, $sHoraFinal,
        $script, $id_url)
    {

        $bExito = FALSE;

        $sNombre = trim($sNombre);
        $iMaxCanales = trim($iMaxCanales);
        $iRetries = trim($iRetries);
        $sTrunk = trim($sTrunk);
        $sContext = trim($sContext);
        $sQueue = trim($sQueue);
        $sFechaInicial = trim($sFechaInicial);
        $sFechaFinal = trim($sFechaFinal);
        $sHoraInicio = trim($sHoraInicio);
        $sHoraFinal = trim($sHoraFinal);
        $script = trim($script);

        if ($sTrunk == '') $sTrunk = NULL;

        if ($sNombre == '') {
            $this->errMsg = _tr("Name Campaign can't be empty");//'Nombre de campaña no puede estar vacío';
        } elseif ($sContext == '') {
            $this->errMsg = _tr("Context can't be empty");//'Contexto no puede estar vacío';
        } elseif (!ctype_digit($iRetries)) {
            $this->errMsg = _tr('Retries must be numeric');//'Número de reintentos debe de ser numérico y entero';
        } elseif ($sQueue == '') {
            $this->errMsg = _tr("Queue can't be empty");//'Número de cola no puede estar vacío';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaInicial)) {
            $this->errMsg = _tr('Invalid Start Date');//'Fecha de inicio no es válida (se espera yyyy-mm-dd)';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaFinal)) {
            $this->errMsg = _tr('Invalid End Date');//'Fecha de final no es válida (se espera yyyy-mm-dd)';
        } elseif ($sFechaInicial > $sFechaFinal) {
            $this->errMsg = _tr('Start Date must be greater than End Date');//'Fecha de inicio debe ser anterior a la fecha final';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraInicio)) {
            $this->errMsg = _tr('Invalid Start Time');//'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraFinal)) {
            $this->errMsg = _tr('Invalid End Time');//'Hora de final no es válida (se espera hh:mm)';
        } elseif (strcmp($sFechaInicial,$sFechaFinal)==0 && strcmp ($sHoraInicio,$sHoraFinal)>=0) {
            $this->errMsg = _tr('Start Time must be greater than End Time');//'Hora de inicio debe ser anterior a la hora final';
        } elseif (!is_null($id_url) && !ctype_digit("$id_url")) {
            $this->errMsg = _tr('(internal) Invalid URL ID');
        } else {

            // Construir y ejecutar la orden de update SQL
            $sPeticionSQL = <<<SQL_UPDATE_CAMPAIGN
UPDATE campaign SET
    name = ?, max_canales = ?, retries = ?, trunk = ?,
    context = ?, queue = ?, datetime_init = ?, datetime_end = ?,
    daytime_init = ?, daytime_end = ?, script = ?, id_url = ?
WHERE id = ?
SQL_UPDATE_CAMPAIGN;
            $paramSQL = array($sNombre, $iMaxCanales, $iRetries, $sTrunk,
                $sContext, $sQueue, $sFechaInicial, $sFechaFinal,
                $sHoraInicio, $sHoraFinal, $script, $id_url,
                $idCampaign);
            if ($this->_DB->genQuery($sPeticionSQL, $paramSQL)) return TRUE;
            $this->errMsg = $this->_DB->errMsg;
        }
        return false;
    }
    
        function updateCampaignNew($idCampaign, $sNombre, $iRetries, $sFechaInicial, $sFechaFinal, $sHoraInicio, $sHoraFinal, $id_num_keyboard, $id_sounds, $max_canales, $break_ini, $break_fin, $active_days = array()) {

        $bExito = FALSE;
 
        $sNombre = trim($sNombre);
       // $iMaxCanales = trim($iMaxCanales);
        $iRetries = trim($iRetries);
       // $sTrunk = trim($sTrunk);
       // $sContext = trim($sContext);
       // $sQueue = trim($sQueue);
        $sFechaInicial = trim($sFechaInicial);
        $sFechaFinal = trim($sFechaFinal);
        $sHoraInicio = trim($sHoraInicio);
        $sHoraFinal = trim($sHoraFinal);
        
        $break_ini = trim($break_ini);
        $break_fin = trim($break_fin);
       // $script = trim($script);

        if ($sTrunk == '')
            $sTrunk = NULL;

        if ($sNombre == '') {
            $this->errMsg = _tr("Name Campaign can't be empty"); //'Nombre de campaña no puede estar vacío';
       } elseif ($id_num_keyboard == '') {
            $this->errMsg = _tr("Numeric Keyboard can't be empty"); //'Contexto no puede estar vacío';
        } elseif (!ctype_digit($iRetries)) {
            $this->errMsg = _tr('Retries must be numeric'); //'Número de reintentos debe de ser numérico y entero';
      //  } elseif ($sQueue == '') {
        //    $this->errMsg = _tr("Queue can't be empty"); //'Número de cola no puede estar vacío';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaInicial)) {
            $this->errMsg = _tr('Invalid Start Date'); //'Fecha de inicio no es válida (se espera yyyy-mm-dd)';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaFinal)) {
            $this->errMsg = _tr('Invalid End Date'); //'Fecha de final no es válida (se espera yyyy-mm-dd)';
        } elseif ($sFechaInicial > $sFechaFinal) {
            $this->errMsg = _tr('Start Date must be greater than End Date'); //'Fecha de inicio debe ser anterior a la fecha final';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraInicio)) {
            $this->errMsg = _tr('Invalid Start Time'); //'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraFinal)) {
            $this->errMsg = _tr('Invalid End Time'); //'Hora de final no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $break_ini)) {
            $this->errMsg = _tr('Invalid Start Time'); //'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $break_fin)) {
            $this->errMsg = _tr('Invalid End Time');
        }elseif (strcmp($sFechaInicial, $sFechaFinal) == 0 && strcmp($sHoraInicio, $sHoraFinal) >= 0 && strcmp($break_ini, $break_fin) >= 0) {
            $this->errMsg = _tr('Start Time must be greater than End Time'); //'Hora de inicio debe ser anterior a la hora final';
        //} elseif (!is_null($id_url) && !ctype_digit("$id_url")) {
          //  $this->errMsg = _tr('(internal) Invalid URL ID');
        } else {

            // Construir y ejecutar la orden de update SQL
            $sPeticionSQL = <<<SQL_UPDATE_CAMPAIGN
UPDATE campaign SET
    name = ?, retries = ?, 
   datetime_init = ?, datetime_end = ?,
    daytime_init = ?, daytime_end = ?, id_num_keyboard = ?, id_sounds = ?, max_canales = ?, breaktime_init = ?, breaktime_end = ? , active_day = ?
WHERE id = ?
SQL_UPDATE_CAMPAIGN;
            $paramSQL = array($sNombre, $iRetries, 
                $sFechaInicial, $sFechaFinal,
                $sHoraInicio, $sHoraFinal,  $id_num_keyboard, $id_sounds, $max_canales, $break_ini, $break_fin, json_encode($active_days), 
                $idCampaign);
            if ($this->_DB->genQuery($sPeticionSQL, $paramSQL))
                return TRUE;
            $this->errMsg = $this->_DB->errMsg;
        }
        return false;
    }

    function activar_campaign($idCampaign, $activar)
    {
        if (!$this->_DB->genQuery(
            'UPDATE campaign SET estatus = ? WHERE id = ?',
            array($activar, $idCampaign))) {
        	$this->errMsg = $this->_DB->errMsg;
            return FALSE;
        }
        return TRUE;
    }

    function delete_campaign($idCampaign) {
        
        $listaSQL = array(
            //  'DELETE FROM campaign_form WHERE id_campaign = ?',
             'DELETE FROM call_recording WHERE id_call_outgoing IN (SELECT id from calls WHERE id_campaign = ?)',
            'DELETE FROM call_attribute WHERE id_call IN (SELECT id from calls WHERE id_campaign = ?)',
            // 'DELETE FROM form_data_recolected WHERE id_calls IN (SELECT id from calls WHERE id_campaign = ?)',
            'DELETE call_progress_log FROM call_progress_log, calls ' .
            'WHERE call_progress_log.id_call_outgoing = calls.id AND calls.id_campaign = ?',
            'DELETE FROM calls WHERE id_campaign = ?',
            'DELETE FROM campaign WHERE id = ?',
        );

        $this->_DB->beginTransaction();
        foreach ($listaSQL as $sql) {
            $r = $this->_DB->genQuery($sql, array($idCampaign));
            if (!$r) {
                $this->errMsg = $this->_DB->errMsg;
                $this->_DB->rollBack();
                return FALSE;
            }
        }
        $this->_DB->commit();
        return TRUE;
    }

    /**
     * Procedimiento para leer la totalidad de los datos de una campaña terminada,
     * incluyendo todos los datos recogidos en los diversos formularios asociados.
     *
     * @param   object  $pDB            Conexión paloDB a la base de datos call_center
     * @param   int     $id_campaign    ID de la campaña a recuperar
     * @param(out) string $errMsg       Mensaje de error
     *
     * @return  NULL en caso de error, o una estructura de la siguiente forma:
    array(
        BASE => array(
            LABEL   =>  array(
                "id_call",
                "Phone Customer"
                ...
            ),
            DATA    =>  array(
                array(...),
                array(...),
                ...
            ),
        ),
        FORMS => array(
            {id_form} => array(
                NAME    =>  'TestForm',
                LABEL   =>  array(
                    "Label A",
                    "Label B"
                    ...
                ),
                DATA    =>  array(
                    {id_call} => array(...),
                    {id_call} => array(...),
                    ...
                ),
            ),
            ...
        ),
    )
     */
    function & getCompletedCampaignData($id_campaign)
    {

        $this->errMsg = NULL;

        $sqlLlamadas = <<<SQL_LLAMADAS
SELECT
    c.id                AS id,
    c.phone             AS telefono,
    c.status            AS estado,
    c.start_time        AS fecha_hora,
    c.duration          AS duracion,
    c.uniqueid          AS uniqueid,
    c.failure_cause     AS failure_cause,
    c.failure_cause_txt AS failure_cause_txt
FROM calls c
WHERE
    c.id_campaign = ? AND
    (c.status='Success' OR c.status='Failure' OR c.status='ShortCall' OR c.status='NoAnswer' OR c.status='Abandoned')
ORDER BY
    telefono ASC
SQL_LLAMADAS;

        $datosCampania = NULL;
        $datosTelefonos = $this->_DB->fetchTable($sqlLlamadas, FALSE, array($id_campaign));
        if (!is_array($datosTelefonos)) {
            $this->errMsg = 'Unable to read campaign phone data - '.$this->_DB->errMsg;
            return $datosCampania;
        }
        $datosCampania = array(
            'BASE'  =>  array(
                'LABEL' =>  array(
                    'id_call',
                    _tr('Phone Customer'),
                    _tr('Status Call'),
                    _tr('Date & Time'),
                    _tr('Duration'),
                    'Uniqueid',
                    _tr('Failure Code'),
                    _tr('Failure Cause'),
                ),
                'DATA'  =>  $datosTelefonos,
            ),
            'FORMS' =>  array(),
        );
        $datosTelefonos = NULL;

        // Construir índice para obtener la posición de la llamada, dado su ID
        $datosCampania['BASE']['ID2POS'] = array();
        foreach ($datosCampania['BASE']['DATA'] as $pos => $tuplaTelefono) {
            $datosCampania['BASE']['ID2POS'][$tuplaTelefono[0]] = $pos;
        }

        // Leer los datos de los atributos de cada llamada
        $iOffsetAttr = count($datosCampania['BASE']['LABEL']);
        $sqlAtributos = <<<SQL_ATRIBUTOS
SELECT
    call_attribute.id_call          AS id_call,
    call_attribute.columna          AS etiqueta,
    call_attribute.value            AS valor,
    call_attribute.column_number    AS posicion
FROM calls, call_attribute
WHERE calls.id_campaign = ? AND calls.id = ? AND calls.id = call_attribute.id_call AND
    (calls.status='Success' OR calls.status='Failure' OR calls.status='ShortCall' OR calls.status='NoAnswer' OR calls.status='Abandoned')
ORDER BY calls.id, call_attribute.column_number
SQL_ATRIBUTOS;
        foreach ($datosCampania['BASE']['ID2POS'] as $id_call => $pos) {
            $datosAtributos = $this->_DB->fetchTable($sqlAtributos, TRUE, array($id_campaign, $id_call));
            if (!is_array($datosAtributos)) {
                $this->errMsg = 'Unable to read attribute data - '.$this->_DB->errMsg;
                $datosCampania = NULL;
                return $datosCampania;
            }
            foreach ($datosAtributos as $tuplaAtributo) {
                // Se asume que el valor posicion empieza desde 1
                $iPos = $iOffsetAttr + $tuplaAtributo['posicion'] - 1;
                $datosCampania['BASE']['LABEL'][$iPos] = $tuplaAtributo['etiqueta'];
                $datosCampania['BASE']['DATA'][$pos][$iPos] = $tuplaAtributo['valor'];
            }
        }

        // Leer los datos de los formularios asociados a esta campaña
/*
        $sqlFormularios = <<<SQL_FORMULARIOS
(SELECT
    f.id        AS id_form,
    ff.id       AS id_form_field,
    ff.etiqueta AS campo_nombre,
    f.nombre    AS formulario_nombre,
    ff.orden    AS orden
FROM campaign_form cf, form f, form_field ff
WHERE cf.id_form = f.id AND f.id = ff.id_form AND ff.tipo <> 'LABEL' AND cf.id_campaign = ?)
UNION DISTINCT
(SELECT DISTINCT
    f.id        AS id_form,
    ff.id       AS id_form_field,
    ff.etiqueta AS campo_nombre,
    f.nombre    AS formulario_nombre,
    ff.orden    AS orden
FROM form f, form_field ff, form_data_recolected fdr, calls c
WHERE f.id = ff.id_form AND ff.tipo <> 'LABEL' AND fdr.id_form_field = ff.id AND fdr.id_calls = c.id AND c.id_campaign = ?)
ORDER BY id_form, orden ASC
SQL_FORMULARIOS;
        $datosFormularios = $this->_DB->fetchTable($sqlFormularios, FALSE, array($id_campaign, $id_campaign));
        if (!is_array($datosFormularios)) {
            $this->errMsg = 'Unable to read form data - '.$this->_DB->errMsg;
            $datosCampania = NULL;
            return $datosCampania;
        }
        foreach ($datosFormularios as $tuplaFormulario) {
            if (!isset($datosCampania['FORMS'][$tuplaFormulario[0]])) {
                $datosCampania['FORMS'][$tuplaFormulario[0]] = array(
                    'NAME'  =>  $tuplaFormulario[3],
                    'LABEL' =>  array(),
                    'DATA'  =>  array(),
                    'FF2POS'=>  array(),
                );
            }
            $datosCampania['FORMS'][$tuplaFormulario[0]]['LABEL'][] = $tuplaFormulario[2];

            // Construir índice para obtener posición/orden del campo de formulario, dado su ID.
            $datosCampania['FORMS'][$tuplaFormulario[0]]['FF2POS'][$tuplaFormulario[1]] = count($datosCampania['FORMS'][$tuplaFormulario[0]]['LABEL']) - 1;
        }
        $datosFormularios = NULL;

        // Leer los datos recolectados de los formularios
        $sqlDatosForm = <<<SQL_DATOS_FORM
SELECT
    c.id AS id_call,
    ff.id_form AS id_form,
    ff.id AS id_form_field,
    fdr.value AS campo_valor
FROM calls c, form_data_recolected fdr, form_field ff
WHERE fdr.id_calls = c.id AND fdr.id_form_field = ff.id AND c.id_campaign = ?
    AND ff.tipo <> 'LABEL'
    AND (c.status='Success' OR c.status='Failure' OR c.status='ShortCall' OR c.status='NoAnswer' OR c.status='Abandoned')
ORDER BY id_call, id_form, id_form_field
SQL_DATOS_FORM;
        $datosRecolectados = $this->_DB->fetchTable($sqlDatosForm, TRUE, array($id_campaign));
        if (!is_array($datosRecolectados)) {
            $this->errMsg = 'Unable to read form fill-out data - '.$this->_DB->errMsg;
            $datosCampania = NULL;
            return $datosCampania;
        }
        foreach ($datosRecolectados as $vr) {
            if (!isset($datosCampania['FORMS'][$vr['id_form']]['DATA'][$vr['id_call']])) {
                // No está asignada la tupla de valores para esta llamada. Se construye
                // una tupla de valores NULL que será llenada progresivamente.
                $tuplaVacia = array_fill(0, count($datosCampania['FORMS'][$vr['id_form']]['LABEL']), NULL);
                $datosCampania['FORMS'][$vr['id_form']]['DATA'][$vr['id_call']] = $tuplaVacia;
            }
            $iPos = $datosCampania['FORMS'][$vr['id_form']]['FF2POS'][$vr['id_form_field']];
            $datosCampania['FORMS'][$vr['id_form']]['DATA'][$vr['id_call']][$iPos] = $vr['campo_valor'];
        }
        $datosRecolectados = NULL;
*/
        return $datosCampania;
    }
    
     public function generate_file($id_campaign) {

        $sqlFile = 'SELECT c.context, nk.number, nk.value, nk.type, s.name_internal '
                . 'FROM campaign c '
                . 'LEFT JOIN numeric_keyboard  nkb ON nkb.id = c.id_num_keyboard '
                . 'LEFT JOIN numeric_key nk ON nk.id_keyboard = nkb.id '
                . 'LEFT JOIN audios s ON s.id= c.id_sounds'
        ;

        $contexts = array();

        $data = $this->_DB->fetchTable($sqlFile);
        $s = '';

        // echo("<pre>");
        // print_r($data); 
        //exit;

        foreach ($data as $val) {

            // $sound_name = $val[4];
            $contexts[$val[0]]["sound"] = $val[4];

            if ($val[3] == 'TEXT')
                continue;

            if ($val[3] == 'LIST') {

                $_val = explode(',', $val[2]);

                foreach ($_val as $k => $v)
                    if (empty($v))
                        unset($_val[$k]);

                $_val = implode('-', $_val);
                $contexts[$val[0]][(int)$val[1]] = $_val;
            }
            if ($val[3] == 'LABEL')
                $contexts[$val[0]][(int)$val[1]] = 'TC';

            if ($val[3] == 'LABEL_SK')
                $contexts[$val[0]][(int)$val[1]] = 'SK';
        }
        
        // echo("<pre>");
        // print_r($contexts); 
        // exit;
            foreach ($contexts as $campaign => $keys) {

                // echo("<pre>");
                $msg = $keys["sound"];
                unset($keys["sound"]);
                ksort($keys);

                // print_r($keys); 
                // $msg = array_pop($keys);  
                // exit;

                $s .= "[$campaign]" . "\n\n";
                $s .= 'exten => s,1,Set(TIMEOUT_LOOPCOUNT=0)' . "\n";
                $s .= 'exten => s,n,Set(INVALID_LOOPCOUNT=0)' . "\n";
                $s .= 'exten => s,n,NoOp(${CONTEXT})' . "\n";
                $s .= 'exten => s,n,Set(_IVR_CONTEXT_${CONTEXT}=${IVR_CONTEXT})' . "\n";
                $s .= 'exten => s,n,Set(_IVR_CONTEXT=${CONTEXT})' . "\n";
                $s .= 'exten => s,n,Set(__IVR_RETVM=)' . "\n";
                $s .= 'exten => s,n,GotoIf($["${CDR(disposition)}" = "ANSWERED"]?skip)' . "\n";
                $s .= 'exten => s,n,Answer' . "\n";
                $s .= 'exten => s,n,Wait(1)' . "\n";
                $s .= "exten => s,n(skip),Set(IVR_MSG=$msg)" . "\n";
                $s .= 'exten => s,n(start),Set(TIMEOUT(digit)=3)' . "\n";
                $s .= 'exten => s,n,ExecIf($["${IVR_MSG}" != ""]?Background(campaign/${IVR_MSG}))' . "\n";
                $s .= 'exten => s,n,WaitExten(10,)' . "\n";

                $s .= "\n\n";

                foreach ($keys as $key => $val) {

                    if ($val != 'TC' && $val != 'SK') {

                        // if($key == "sound" ) continue;

                        $s .= "exten => $key,1,NoOp(RG Campanha)" . "\n";
                        $s .= "exten => $key,n,Set(RG_LOOPCOUNT=0)" . "\n";
                        $s .= "exten => $key,n(rgstart),Set(__IVR_DIGIT_PRESSED=$key)" . "\n";

                        $s .= "exten => $key,n,Gosub(sub-record-check,s,1(rg,1600,always))" . "\n";

                        $s .= "exten => $key,n,Set(RingGroupMethod=ringall)" . "\n";
                        $s .= 'exten => ' . $key . ',n(DIALGRP),Macro(dial,20,${DIAL_OPTIONS},' . $val . ')' . "\n";

                        $s .= "exten => $key,n,Gosub(sub-record-cancel,s,1())" . "\n";

                        $s .= 'exten => ' . $key . ',n,Set(RG_LOOPCOUNT=$[${RG_LOOPCOUNT}+1])' . "\n";
                        $s .= "exten => $key,n,Set(RingGroupMethod=)" . "\n";
                        $s .= 'exten => ' . $key . ',n,GotoIf($[${RG_LOOPCOUNT} > 2]?final)' . "\n";
                        $s .= "exten => $key,n,Goto($key,rgstart)" . "\n";
                        $s .= "exten => $key,n(final),AGI(campaign_audio.agi)" . "\n";
                        $s .= "exten => $key,n,Playback(campaign/rg-congestion)" . "\n";
                        $s .= "exten => $key,n,Goto(app-blackhole,congestion,1)" . "\n\n";
                    }

                    if ($val == 'TC') {
                        $s .= "exten => $key,1,Set(__IVR_DIGIT_PRESSED=$key)" . "\n";
                        $s .= "exten => $key,n(ivrsel-$key),Goto(app-blackhole,hangup,1)" . "\n\n";
                    }

                    if ($val == 'SK') {
                        $s .= "exten => $key,1,NoOp(Storage Key)" . "\n";
                        $s .= 'exten => '.$key.',n,Set(STORAGE_KEY=${EXTEN})' . "\n";
                        $s .= "exten => $key,n,AGI(campaign_audio-storage_key.agi)" . "\n";
			$s .= "exten => $key,n,Hangup(16)" . "\n\n";
			
                    }
                }

                $s .= "\n\n";

                $s .= 'exten => i,1,Set(INVALID_LOOPCOUNT=$[${INVALID_LOOPCOUNT}+1])' . "\n";
                $s .= 'exten => i,n,GotoIf($[${INVALID_LOOPCOUNT} > 0]?final)' . "\n";
                $s .= 'exten => i,n,Set(IVR_MSG=ate-logo)' . "\n";
                $s .= 'exten => i,n,Goto(s,start)' . "\n";
                $s .= 'exten => i,n(final),Playback(campaign/ate-logo)' . "\n";
                $s .= 'exten => i,n,Goto(app-blackhole,hangup,1)' . "\n";

                $s .= "\n\n";

                $s .= 'exten => t,1,Set(TIMEOUT_LOOPCOUNT=$[${TIMEOUT_LOOPCOUNT}+1])' . "\n";
                $s .= 'exten => t,n,GotoIf($[${TIMEOUT_LOOPCOUNT} > 0]?final)' . "\n";
                $s .= 'exten => t,n,Set(IVR_MSG=ate-logo)' . "\n";
                $s .= 'exten => t,n,Goto(s,start)' . "\n";
                $s .= 'exten => t,n(final),Playback(campaign/ate-logo)' . "\n";
                $s .= 'exten => t,n,Goto(app-blackhole,hangup,1)' . "\n";

                $s .= "\n\n";

                $s .= 'exten => return,1,Set(_IVR_CONTEXT=${CONTEXT})' . "\n";
                $s .= 'exten => return,n,Set(_IVR_CONTEXT_${CONTEXT}=${IVR_CONTEXT_${CONTEXT}})' . "\n";
                $s .= 'exten => return,n,Set(IVR_MSG=)' . "\n";
                $s .= 'exten => return,n,Goto(s,start)' . "\n";


                $s .= "\n\n";

                $s .= 'exten => h,1,Hangup' . "\n";

                $s .= "\n\n";

                $s .= 'exten => hang,1,Playback(vm-goodbye)' . "\n";
                $s .= 'exten => hang,n,Hangup' . "\n\n\n";
            }
        
        
        $filename = '/etc/asterisk/extensions_campaign.conf';

        $fd = fopen($filename, "w+");

        fwrite($fd, $s);

        fclose($fd);
    } 

    function repeatCallcenter($id_campaign, $sql_in, $phone_type) {


        if (in_array('"NoCall"', $sql_in)) {
            $is_null = ' OR cl.status IS NULL ';
        } else
            $is_null = '';


        $in = implode(',', $sql_in);

        $sqlRepeat = <<<SQL_REPEAT
SELECT
    cl.uuid,
    cl.phone, 
    cat.columna, 
    cat.value
FROM  campaign_audio.campaign as c
INNER JOIN campaign_audio.calls cl 
    ON cl.id_campaign = c.id
INNER JOIN campaign_audio.call_attribute cat
    ON cat.id_call = cl.id 
WHERE
    status IN ($in) AND
    c.id = ? 
              
SQL_REPEAT;

        $data = $this->_DB->fetchTable($sqlRepeat, TRUE, array($id_campaign));

        $line = array();
        $colunm = array();

        foreach ($data as $tupla) {
            
            if (!array_key_exists($tupla['uuid'], $line)) {
             
                $line[$tupla['uuid']] = array();
            }

            if (!array_key_exists($tupla['phone'], $line[$tupla['uuid']])) {

                // Tirar zero
                if (substr($tupla['phone'], 0, 1) != '0')
                    $tupla['phone'] = "0" . $tupla['phone'];

                if ($phone_type == 'fix') {

                    if ($this->fix_phone($tupla['phone']))
                        $line [$tupla['uuid']][$tupla['phone']] = array();
                    else
                        continue;
                    
                } else if ($phone_type == 'movel') {

                    if ($this->movel_phone($tupla['phone']))
                        $line [$tupla['uuid']][$tupla['phone']] = array();
                    else
                        continue;
                    
                } else {
                    $line [$tupla['uuid']][$tupla['phone']] = array();
                }
            }
            
            if (!in_array($tupla['columna'], $colunm))
                $colunm [] = $tupla['columna'];

           $line [$tupla['uuid']][$tupla['phone']] [$tupla['columna']] = $tupla['value'];
            
        }
        
        // Lista sem contatos
        if ( count($line) == 0 ) return false;

        $output = '"UUID","DNC","FONE",' . implode(',', array_map('csv_replace', $colunm)) . "\r\n";

        
        foreach ($line as $key => $value) {

            $dnc = 0;
            
            foreach($value as $kfone => $vfone) {
                $output .= '"' . $key . '",' . '"' . $dnc. '",' . '"' . $kfone . '",';
                $output .= implode(',', array_map('csv_replace', $vfone)) . "\r\n";
                $dnc++;
            }
            
        }

        $name_out = '/tmp/' . time() . "_repeat.csv";
        $handle_out = fopen($name_out, "wb");

        fwrite($handle_out, $output);
        fclose($handle_out);

        $sqlCampaign = <<<SQL_CAMPAIGN
SELECT
    name
FROM  campaign_audio.campaign 

WHERE
    id = ? 
LIMIT 1
              
SQL_CAMPAIGN;
        
        

        $data = $this->_DB->fetchTable($sqlCampaign, TRUE, array($id_campaign)); 

        $data = $data[0];
        
        
        $sqlCampaignDefault = <<<SQL_CAMPAIGN_DEFAULT
SELECT
    config_key, config_value 
FROM  valor_config 

WHERE
    config_key LIKE 'campaign.%'  

              
SQL_CAMPAIGN_DEFAULT;
        
        
        $data_config = $this->_DB->fetchTable($sqlCampaignDefault, TRUE, array());
        
        foreach ($data_config as $cnf ) {
            
            switch ($cnf['config_key']) {
                
                case 'campaign.campaign_standard';
                    $data['active'] = $cnf['config_value'];
                    break;
                
                case 'campaign.fecha_fin';
                    $data['datetime_end'] = $cnf['config_value'];
                    break;
                
                case 'campaign.fecha_ini';
                    $data['datetime_init'] = $cnf['config_value'];
                    break;
                
                case 'campaign.hora_fin_HH';
                    $data['daytime_end_HH'] = $cnf['config_value'];
                    break;
                
                case 'campaign.hora_fin_MM';
                    $data['daytime_end_MM'] = $cnf['config_value'];
                    break;
                
                case 'campaign.hora_ini_HH';
                    $data['daytime_init_HH'] = $cnf['config_value'];
                    break;
                
                case 'campaign.hora_ini_MM';
                    $data['daytime_init_MM'] = $cnf['config_value'];
                    break;
                
                case 'campaign.integration';
                    $data['integration'] = $cnf['config_value'];
                    break;
                
                case 'campaign.queue';
                    $data['queue'] = $cnf['config_value'];
                    break;
                
                case 'campaign.reintentos';
                    $data['retries'] = $cnf['config_value'];
                    break;
                
            }
            
        }
        
        $data['max_canales'] = 0;
        $data['trunk'] = '';
        $data['context'] = 'from-internal';
        $data['daytime_init'] = $data['daytime_init_HH']  . ':' . $data['daytime_init_MM'];
        $data['daytime_end'] = $data['daytime_end_HH']  . ':' . $data['daytime_end_MM'];
        $data['script'] = '';
        $data['id_url'] = NULL;
       

        $id_new = $this->createEmptyCampaignCallcenter('(Cópia Ura) ' . $data['name'], $data['max_canales'], $data['retries'], $data['trunk'], $data['context'], $data['queue'], 
                date("Y-m-d",  strtotime($data['datetime_init'])) , date ("Y-m-d", strtotime($data['datetime_end'])), $data['daytime_init'], $data['daytime_end'], $data['script'], $data['id_url'], $data['integration']
        );

        if ($id_new) {
            require_once "modules/campaign_out/libs/paloContactInsert.class.php";
            require_once "modules/campaign_out/uploaders/Storage/index.php";

            Uploader_Storage::addCampaignNumbersFromFileRepeat($this->_DB, $id_new, $name_out, NULL);
        }

        unlink($name_out);
       
        return true;
    }

    function repeatUra($id_campaign, $sql_in, $phone_type) {


        if (in_array('"NoCall"', $sql_in)) {
            $is_null = ' OR cl.status IS NULL ';
        } else
            $is_null = '';


        $in = implode(',', $sql_in);

       $sqlRepeat = <<<SQL_REPEAT
SELECT
    cl.uuid,
    cl.phone, 
    cat.columna, 
    cat.value
FROM  campaign as c
INNER JOIN calls cl 
    ON cl.id_campaign = c.id
INNER JOIN call_attribute cat
    ON cat.id_call = cl.id 
WHERE
    status IN ($in) AND
    c.id = ? 
              
SQL_REPEAT;

        $data = $this->_DB->fetchTable($sqlRepeat, TRUE, array($id_campaign));

        $line = array();
        $colunm = array();

        foreach ($data as $tupla) {
            
            if (!array_key_exists($tupla['uuid'], $line)) {
             
                $line[$tupla['uuid']] = array();
            }

            if (!array_key_exists($tupla['phone'], $line[$tupla['uuid']])) {

                // Tirar zero
                if (substr($tupla['phone'], 0, 1) != '0')
                    $tupla['phone'] = "0" . $tupla['phone'];

                if ($phone_type == 'fix') {

                    if ($this->fix_phone($tupla['phone']))
                        $line [$tupla['uuid']][$tupla['phone']] = array();
                    else
                        continue;
                    
                } else if ($phone_type == 'movel') {

                    if ($this->movel_phone($tupla['phone']))
                        $line [$tupla['uuid']][$tupla['phone']] = array();
                    else
                        continue;
                    
                } else {
                    $line [$tupla['uuid']][$tupla['phone']] = array();
                }
            }
            
            if (!in_array($tupla['columna'], $colunm))
                $colunm [] = $tupla['columna'];

           $line [$tupla['uuid']][$tupla['phone']] [$tupla['columna']] = $tupla['value'];
            
        }
        
        // Lista sem contatos
        if ( count($line) == 0 ) return false;

        $output = '"UUID","DNC","FONE",' . implode(',', array_map('csv_replace', $colunm)) . "\r\n";

        
        foreach ($line as $key => $value) {

            $dnc = 0;
            
            foreach($value as $kfone => $vfone) {
                $output .= '"' . $key . '",' . '"' . $dnc. '",' . '"' . $kfone . '",';
                $output .= implode(',', array_map('csv_replace', $vfone)) . "\r\n";
                $dnc++;
            }
            
        }

        $name_out = '/tmp/' . time() . "_repeat.csv";
        $handle_out = fopen($name_out, "wb");

        fwrite($handle_out, $output);
        fclose($handle_out);

        $sqlCampaign = <<<SQL_CAMPAIGN
SELECT
    *
FROM  campaign 

WHERE
    id = ? 
LIMIT 1
              
SQL_CAMPAIGN;

        $data = $this->_DB->fetchTable($sqlCampaign, TRUE, array($id_campaign));

        $data = $data[0];

        $id_new = $this->createEmptyCampaignNew('Cópia - ' . $data['name'], $data['retries'], $data['context'], $data['datetime_init'], $data['datetime_end'], substr($data['daytime_init'], 0, 5), substr($data['daytime_end'], 0, 5), '', '', $data['max_canales'], '12:00', '13:00', '{"Sun":"off","Mon":"off","Tue":"off","Wed":"off","Thu":"off","Fri":"off","Sat":"off"}'
        );

        if ($id_new) {

            require_once "modules/campaign_rg/libs/paloContactInsert.class.php";
            require_once "modules/campaign_rg/uploaders/Storage/index.php";

            Uploader_Storage::addCampaignNumbersFromFileRepeat($this->_DB, $id_new, $name_out, NULL);
        }

        unlink($name_out);

        return true;
    }
    
    function movel_phone($telefone) {
        $telefone = trim(str_replace('/', '', str_replace(' ', '', str_replace('-', '', str_replace(')', '', str_replace('(', '', $telefone))))));

       // $regexTelefone = "^[0-9]{11}$";

        $regexCel = '/0[1-9]{2}[6789][0-9]{8}/'; // Regex para validar somente celular
        if (preg_match($regexCel, $telefone)) {
            return true;
        } else {
            return false;
        }
    }
    
    function fix_phone($telefone) {
        $telefone = trim(str_replace('/', '', str_replace(' ', '', str_replace('-', '', str_replace(')', '', str_replace('(', '', $telefone))))));

       // $regexTelefone = "^[0-9]{11}$";

        $regexCel = '/0[1-9]{2}[2345][0-9]{7}/'; // Regex para validar somente celular
        if (preg_match($regexCel, $telefone)) {
            return true;
        } else {
            return false;
        }
    }

    function csv_replace($s) {
        return ($s == '') ? '""' : '"' . str_replace('"', "'", $s) . '"';
    }
    
    
     function createEmptyCampaignCallcenter($sNombre, $iMaxCanales, $iRetries, $sTrunk, $sContext, $sQueue,
        $sFechaInicial, $sFechaFinal, $sHoraInicio, $sHoraFinal, $script, $id_url, $integration)
    { 
        $id_campaign = NULL;
        $bExito = FALSE;

        // Carga de colas entrantes activas
        $recordset = $this->_DB->fetchTable("SELECT queue FROM queue_call_entry WHERE estatus='A'");
        if (!is_array($recordset)) {
            $this->errMsg = _tr('(internal) Failed to query active incoming queues').
                ' - '.$this->_DB->errMsg;
        	return NULL;
        }
        $colasEntrantes = array();
        foreach ($recordset as $tupla) $colasEntrantes[] = $tupla[0];

        $sNombre = trim($sNombre);
        $iMaxCanales = trim($iMaxCanales);
        $iRetries = trim($iRetries);
        $sTrunk = trim($sTrunk);
        $sContext = trim($sContext);
        $sQueue = trim($sQueue);
        $sFechaInicial = trim($sFechaInicial);
        $sFechaFinal = trim($sFechaFinal);
        $sHoraInicio = trim($sHoraInicio);
        $sHoraFinal = trim($sHoraFinal);
        $script = trim($script);

        if ($sTrunk == '') $sTrunk = NULL;

        if ($sNombre == '') {
            $this->errMsg = _tr("Name Campaign can't be empty");//'Nombre de campaña no puede estar vacío';
        } elseif ($sContext == '') {
            $this->errMsg = _tr("Context can't be empty");//'Contexto no puede estar vacío';
        } elseif (!ctype_digit($iRetries)) { 
            $this->errMsg = _tr('Retries must be numeric');//'Número de reintentos debe de ser numérico y entero';
        } elseif ($sQueue == '') {  
            $this->errMsg = _tr("Queue can't be empty");//'Número de cola no puede estar vacío';
        } elseif (!ctype_digit($sQueue)) { 
            $this->errMsg = _tr('Queue must be numeric');//'Número de cola debe de ser numérico y entero';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaInicial)) {
            $this->errMsg = _tr('Invalid Start Date');//'Fecha de inicio no es válida (se espera yyyy-mm-dd)';
        } elseif (!preg_match(REGEXP_FECHA_VALIDA, $sFechaFinal)) { 
            $this->errMsg = _tr('Invalid End Date');//'Fecha de final no es válida (se espera yyyy-mm-dd)';
        } elseif ($sFechaInicial > $sFechaFinal) { 
            $this->errMsg = _tr('Start Date must be greater than End Date');//'Fecha de inicio debe ser anterior a la fecha final';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraInicio)) { 
            $this->errMsg = _tr('Invalid Start Time');//'Hora de inicio no es válida (se espera hh:mm)';
        } elseif (!preg_match(REGEXP_HORA_VALIDA, $sHoraFinal)) { 
            $this->errMsg = _tr('Invalid End Time');//'Hora de final no es válida (se espera hh:mm)';
        } elseif (strcmp($sFechaInicial,$sFechaFinal)==0 && strcmp ($sHoraInicio,$sHoraFinal)>=0) { 
            $this->errMsg = _tr('Start Time must be greater than End Time');//'Hora de inicio debe ser anterior a la hora final';
        } elseif (!is_null($id_url) && !ctype_digit("$id_url")) { 
            $this->errMsg = _tr('(internal) Invalid URL ID');
        } elseif (in_array($sQueue, $colasEntrantes)) { 
             $this->errMsg =  _tr('Queue is being used, choose other one');//La cola ya está siendo usada, escoja otra
        } else {   
            // Verificar que el nombre de la campaña es único
            $tupla = $this->_DB->getFirstRowQuery(
                'SELECT COUNT(*) AS N FROM campaign WHERE name = ?', TRUE, array($sNombre));
            if (is_array($tupla) && $tupla['N'] > 0) { 
                // Ya existe una campaña duplicada 
                $this->errMsg = _tr('Name Campaign already exists');//'Nombre de campaña indicado ya está en uso';
            	return NULL;
            }

            // Construir y ejecutar la orden de inserción SQL
            $sPeticionSQL = <<<SQL_INSERT_CAMPAIGN
INSERT INTO campaign (name, max_canales, retries, trunk, context, queue,
    datetime_init, datetime_end, daytime_init, daytime_end, script, id_url, integration, estatus)
VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
SQL_INSERT_CAMPAIGN;
            $paramSQL = array($sNombre, $iMaxCanales, $iRetries, $sTrunk,
                $sContext, $sQueue, $sFechaInicial, $sFechaFinal, $sHoraInicio,
                $sHoraFinal, $script, $id_url, $integration, 'I');
            if ($this->_DB->genQuery($sPeticionSQL, $paramSQL)) {
            	// Leer el ID insertado por la operación
                $id_campaign = $this->_DB->getLastInsertId();
                if ($id_campaign === FALSE) {
                	$this->errMsg = $this->_DB->errMsg;
                    $id_campaign = NULL;
                }
            } else {
            	$this->errMsg = $this->_DB->errMsg;
            }
        }
        return $id_campaign;
    }

    
}

